/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss;

import eu.europa.esig.dss.CertificatePolicy;
import eu.europa.esig.dss.DSSException;
import eu.europa.esig.dss.DSSUtils;
import eu.europa.esig.dss.DigestAlgorithm;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.x509.CertificateToken;
import java.io.IOException;
import java.math.BigInteger;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1GeneralizedTime;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1String;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.DLSet;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.ocsp.BasicOCSPResponse;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.AccessDescription;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.AuthorityInformationAccess;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.PolicyQualifierId;
import org.bouncycastle.asn1.x509.PolicyQualifierInfo;
import org.bouncycastle.asn1.x509.SubjectKeyIdentifier;
import org.bouncycastle.asn1.x509.Time;
import org.bouncycastle.asn1.x509.X509ObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.QCStatement;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.tsp.TimeStampToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DSSASN1Utils {
    private static final Logger LOG = LoggerFactory.getLogger(DSSASN1Utils.class);
    private static final String QC_TYPE_STATEMENT_OID = "0.4.0.1862.1.6";
    private static final BouncyCastleProvider securityProvider = new BouncyCastleProvider();

    private DSSASN1Utils() {
    }

    public static <T extends ASN1Primitive> T toASN1Primitive(byte[] bytes) {
        try {
            return (T)ASN1Primitive.fromByteArray(bytes);
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    private static boolean isDEROctetStringNull(DEROctetString derOctetString) {
        byte[] derOctetStringBytes = derOctetString.getOctets();
        Object asn1Null = DSSASN1Utils.toASN1Primitive(derOctetStringBytes);
        return DERNull.INSTANCE.equals(asn1Null);
    }

    public static byte[] getDEREncoded(ASN1Encodable asn1Encodable) {
        return DSSASN1Utils.getEncoded(asn1Encodable, "DER");
    }

    public static byte[] getBEREncoded(ASN1Encodable asn1Encodable) {
        return DSSASN1Utils.getEncoded(asn1Encodable, "BER");
    }

    private static byte[] getEncoded(ASN1Encodable asn1Encodable, String encoding) {
        try {
            return asn1Encodable.toASN1Primitive().getEncoded(encoding);
        }
        catch (IOException e) {
            throw new DSSException("Unable to encode to " + encoding, e);
        }
    }

    public static byte[] getEncoded(BasicOCSPResp basicOCSPResp) {
        try {
            BasicOCSPResponse basicOCSPResponse = BasicOCSPResponse.getInstance(basicOCSPResp.getEncoded());
            return DSSASN1Utils.getDEREncoded(basicOCSPResponse);
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    public static Date toDate(ASN1GeneralizedTime asn1Date) {
        try {
            return asn1Date.getDate();
        }
        catch (ParseException e) {
            throw new DSSException(e);
        }
    }

    public static String toString(ASN1OctetString value) {
        return new String(value.getOctets());
    }

    public static byte[] getEncoded(TimeStampToken timeStampToken) {
        try {
            return timeStampToken.getEncoded();
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    public static ASN1Sequence getAsn1SequenceFromDerOctetString(byte[] bytes) {
        return DSSASN1Utils.getASN1Sequence(DSSASN1Utils.getDEROctetStringContent(bytes));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static byte[] getDEROctetStringContent(byte[] bytes) {
        try (ASN1InputStream input = new ASN1InputStream(bytes);){
            DEROctetString s = (DEROctetString)input.readObject();
            byte[] byArray = s.getOctets();
            return byArray;
        }
        catch (IOException e) {
            throw new DSSException("Unable to retrieve the DEROctetString content", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static ASN1Sequence getASN1Sequence(byte[] bytes) {
        try (ASN1InputStream input = new ASN1InputStream(bytes);){
            ASN1Sequence aSN1Sequence = (ASN1Sequence)input.readObject();
            return aSN1Sequence;
        }
        catch (IOException e) {
            throw new DSSException("Unable to retrieve the ASN1Sequence", e);
        }
    }

    public static byte[] getAsn1SignaturePolicyDigest(DigestAlgorithm digestAlgorithm, byte[] policyBytes) {
        ASN1Sequence asn1Seq = (ASN1Sequence)DSSASN1Utils.toASN1Primitive(policyBytes);
        ASN1Sequence signPolicyHashAlgObject = (ASN1Sequence)asn1Seq.getObjectAt(0);
        AlgorithmIdentifier signPolicyHashAlgIdentifier = AlgorithmIdentifier.getInstance(signPolicyHashAlgObject);
        ASN1Sequence signPolicyInfo = (ASN1Sequence)asn1Seq.getObjectAt(1);
        byte[] hashAlgorithmDEREncoded = DSSASN1Utils.getDEREncoded(signPolicyHashAlgIdentifier);
        byte[] signPolicyInfoDEREncoded = DSSASN1Utils.getDEREncoded(signPolicyInfo);
        return DSSUtils.digest(digestAlgorithm, hashAlgorithmDEREncoded, signPolicyInfoDEREncoded);
    }

    public static String getCanonicalizedName(GeneralNames generalNames) {
        GeneralName[] names = generalNames.getNames();
        TreeMap<String, String> treeMap = new TreeMap<String, String>();
        for (GeneralName name : names) {
            String ldapString = String.valueOf(name.getName());
            LOG.debug("ldapString to canonicalize: {} ", (Object)ldapString);
            try {
                LdapName ldapName = new LdapName(ldapString);
                List<Rdn> rdns = ldapName.getRdns();
                for (Rdn rdn : rdns) {
                    treeMap.put(rdn.getType().toLowerCase(), String.valueOf(rdn.getValue()).toLowerCase());
                }
            }
            catch (InvalidNameException e) {
                throw new DSSException(e);
            }
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (Map.Entry entry : treeMap.entrySet()) {
            stringBuilder.append((String)entry.getKey()).append('=').append((String)entry.getValue()).append('|');
        }
        String canonicalizedName = stringBuilder.toString();
        LOG.debug("canonicalizedName: {} ", (Object)canonicalizedName);
        return canonicalizedName;
    }

    public static AlgorithmIdentifier getAlgorithmIdentifier(DigestAlgorithm digestAlgorithm) {
        ASN1ObjectIdentifier asn1ObjectIdentifier = new ASN1ObjectIdentifier(digestAlgorithm.getOid());
        return new AlgorithmIdentifier(asn1ObjectIdentifier, DERNull.INSTANCE);
    }

    public static boolean hasIdPkixOcspNoCheckExtension(CertificateToken token) {
        byte[] extensionValue = token.getCertificate().getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nocheck.getId());
        if (extensionValue != null) {
            try {
                Object derObject = DSSASN1Utils.toASN1Primitive(extensionValue);
                if (derObject instanceof DEROctetString) {
                    return DSSASN1Utils.isDEROctetStringNull((DEROctetString)derObject);
                }
            }
            catch (Exception e) {
                LOG.debug("Exception when processing 'id_pkix_ocsp_no_check'", e);
            }
        }
        return false;
    }

    public static List<CertificatePolicy> getCertificatePolicies(CertificateToken certToken) {
        ArrayList<CertificatePolicy> certificatePolicies = new ArrayList<CertificatePolicy>();
        byte[] certificatePoliciesBinaries = certToken.getCertificate().getExtensionValue(Extension.certificatePolicies.getId());
        if (Utils.isArrayNotEmpty(certificatePoliciesBinaries)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(certificatePoliciesBinaries);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    CertificatePolicy cp = new CertificatePolicy();
                    PolicyInformation policyInfo = PolicyInformation.getInstance(seq.getObjectAt(ii));
                    cp.setOid(policyInfo.getPolicyIdentifier().getId());
                    ASN1Sequence policyQualifiersSeq = policyInfo.getPolicyQualifiers();
                    if (policyQualifiersSeq != null) {
                        for (int jj = 0; jj < policyQualifiersSeq.size(); ++jj) {
                            PolicyQualifierInfo pqi = PolicyQualifierInfo.getInstance(policyQualifiersSeq.getObjectAt(jj));
                            if (!PolicyQualifierId.id_qt_cps.equals(pqi.getPolicyQualifierId())) continue;
                            cp.setCpsUrl(DSSASN1Utils.getString(pqi.getQualifier()));
                        }
                    }
                    certificatePolicies.add(cp);
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the certificatePolicies extension '" + Utils.toBase64(certificatePoliciesBinaries) + "' : " + e.getMessage(), e);
            }
        }
        return certificatePolicies;
    }

    public static List<String> getQCStatementsIdList(CertificateToken certToken) {
        ArrayList<String> extensionIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty(qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    QCStatement statement = QCStatement.getInstance(seq.getObjectAt(ii));
                    extensionIdList.add(statement.getStatementId().getId());
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the qCStatements extension '" + Utils.toBase64(qcStatement) + "' : " + e.getMessage(), e);
            }
        }
        return extensionIdList;
    }

    public static List<String> getQCTypesIdList(CertificateToken certToken) {
        ArrayList<String> qcTypesIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty(qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    QCStatement statement = QCStatement.getInstance(seq.getObjectAt(ii));
                    if (!QC_TYPE_STATEMENT_OID.equals(statement.getStatementId().getId())) continue;
                    ASN1Encodable qcTypeInfo1 = statement.getStatementInfo();
                    if (qcTypeInfo1 instanceof ASN1Sequence) {
                        ASN1Sequence qcTypeInfo = (ASN1Sequence)qcTypeInfo1;
                        for (int jj = 0; jj < qcTypeInfo.size(); ++jj) {
                            ASN1Encodable e1 = qcTypeInfo.getObjectAt(jj);
                            if (e1 instanceof ASN1ObjectIdentifier) {
                                ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)e1;
                                qcTypesIdList.add(oid.getId());
                                continue;
                            }
                            LOG.warn("ASN1Sequence in QcTypes does not contain ASN1ObjectIdentifer, but {}", (Object)e1.getClass().getName());
                        }
                        continue;
                    }
                    LOG.warn("QcTypes not an ASN1Sequence, but {}", (Object)qcTypeInfo1.getClass().getName());
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the qCStatements extension '" + Utils.toBase64(qcStatement) + "' : " + e.getMessage(), e);
            }
        }
        return qcTypesIdList;
    }

    public static byte[] getSki(CertificateToken certificateToken) {
        return DSSASN1Utils.getSki(certificateToken, false);
    }

    public static byte[] getSki(CertificateToken certificateToken, boolean computeIfMissing) {
        try {
            byte[] sKI = certificateToken.getCertificate().getExtensionValue(Extension.subjectKeyIdentifier.getId());
            if (Utils.isArrayNotEmpty(sKI)) {
                ASN1Primitive extension = JcaX509ExtensionUtils.parseExtensionValue(sKI);
                SubjectKeyIdentifier skiBC = SubjectKeyIdentifier.getInstance(extension);
                return skiBC.getKeyIdentifier();
            }
            if (computeIfMissing) {
                return DSSASN1Utils.computeSkiFromCert(certificateToken);
            }
            return null;
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    public static byte[] computeSkiFromCert(CertificateToken certificateToken) {
        try {
            DLSequence seq = (DLSequence)DERSequence.fromByteArray(certificateToken.getPublicKey().getEncoded());
            DERBitString item = (DERBitString)seq.getObjectAt(1);
            return DSSUtils.digest(DigestAlgorithm.SHA1, item.getOctets());
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    public static List<String> getCAAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_caIssuers);
    }

    public static List<String> getOCSPAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_ocsp);
    }

    private static List<String> getAccessLocations(CertificateToken certificate, ASN1ObjectIdentifier aiaType) {
        ArrayList<String> locationsUrls = new ArrayList<String>();
        byte[] authInfoAccessExtensionValue = certificate.getCertificate().getExtensionValue(Extension.authorityInfoAccess.getId());
        if (null == authInfoAccessExtensionValue) {
            return locationsUrls;
        }
        try {
            AccessDescription[] accessDescriptions;
            ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(authInfoAccessExtensionValue);
            AuthorityInformationAccess authorityInformationAccess = AuthorityInformationAccess.getInstance(asn1Sequence);
            for (AccessDescription accessDescription : accessDescriptions = authorityInformationAccess.getAccessDescriptions()) {
                GeneralName gn;
                String location;
                if (!aiaType.equals(accessDescription.getAccessMethod()) || (location = DSSASN1Utils.parseGn(gn = accessDescription.getAccessLocation())) == null) continue;
                locationsUrls.add(location);
            }
        }
        catch (Exception e) {
            LOG.error("Unable to parse authorityInfoAccess", e);
        }
        return locationsUrls;
    }

    public static List<String> getCrlUrls(CertificateToken certificateToken) {
        ArrayList<String> urls = new ArrayList<String>();
        byte[] crlDistributionPointsBytes = certificateToken.getCertificate().getExtensionValue(Extension.cRLDistributionPoints.getId());
        if (crlDistributionPointsBytes != null) {
            try {
                DistributionPoint[] distributionPoints;
                ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(crlDistributionPointsBytes);
                CRLDistPoint distPoint = CRLDistPoint.getInstance(asn1Sequence);
                for (DistributionPoint distributionPoint : distributionPoints = distPoint.getDistributionPoints()) {
                    GeneralName[] names;
                    DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
                    if (0 != distributionPointName.getType()) continue;
                    GeneralNames generalNames = (GeneralNames)distributionPointName.getName();
                    for (GeneralName name : names = generalNames.getNames()) {
                        String location = DSSASN1Utils.parseGn(name);
                        if (location == null) continue;
                        urls.add(location);
                    }
                }
            }
            catch (Exception e) {
                LOG.error("Unable to parse cRLDistributionPoints", e);
            }
        }
        return urls;
    }

    private static String parseGn(GeneralName gn) {
        try {
            if (6 == gn.getTagNo()) {
                ASN1String str = (ASN1String)((Object)((DERTaggedObject)gn.toASN1Primitive()).getObject());
                return str.getString();
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to parse GN " + gn, e);
        }
        return null;
    }

    public static boolean isOCSPSigning(CertificateToken certToken) {
        return DSSASN1Utils.isExtendedKeyUsagePresent(certToken, KeyPurposeId.id_kp_OCSPSigning.toOID());
    }

    public static boolean isExtendedKeyUsagePresent(CertificateToken certToken, ASN1ObjectIdentifier oid) {
        try {
            List<String> keyPurposes = certToken.getCertificate().getExtendedKeyUsage();
            if (keyPurposes != null && keyPurposes.contains(oid.getId())) {
                return true;
            }
        }
        catch (CertificateParsingException e) {
            LOG.error("Unable to retrieve ExtendedKeyUsage from certificate", e);
        }
        return false;
    }

    public static X509CertificateHolder getX509CertificateHolder(CertificateToken certToken) {
        try {
            return new X509CertificateHolder(certToken.getEncoded());
        }
        catch (IOException e) {
            throw new DSSException(e);
        }
    }

    public static CertificateToken getCertificate(X509CertificateHolder x509CertificateHolder) {
        try {
            JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider("BC");
            X509Certificate x509Certificate = converter.getCertificate(x509CertificateHolder);
            return new CertificateToken(x509Certificate);
        }
        catch (CertificateException e) {
            throw new DSSException(e);
        }
    }

    public static IssuerSerial getIssuerSerial(CertificateToken certToken) {
        X500Name issuerX500Name = DSSASN1Utils.getX509CertificateHolder(certToken).getIssuer();
        GeneralName generalName = new GeneralName(issuerX500Name);
        GeneralNames generalNames = new GeneralNames(generalName);
        BigInteger serialNumber = certToken.getCertificate().getSerialNumber();
        return new IssuerSerial(generalNames, serialNumber);
    }

    public static Map<String, String> get(X500Principal x500Principal) {
        ASN1Encodable[] asn1Encodables;
        HashMap<String, String> treeMap = new HashMap<String, String>();
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance(encoded);
        for (ASN1Encodable asn1Encodable : asn1Encodables = asn1Sequence.toArray()) {
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int ii = 0; ii < dlSet.size(); ++ii) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(ii);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable asn1EncodableAttributeType = dlSequence.getObjectAt(0);
                String stringAttributeType = DSSASN1Utils.getString(asn1EncodableAttributeType);
                ASN1Encodable asn1EncodableAttributeValue = dlSequence.getObjectAt(1);
                String stringAttributeValue = DSSASN1Utils.getString(asn1EncodableAttributeValue);
                treeMap.put(stringAttributeType, stringAttributeValue);
            }
        }
        return treeMap;
    }

    public static String getUtf8String(X500Principal x500Principal) {
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance(encoded);
        ASN1Encodable[] asn1Encodables = asn1Sequence.toArray();
        StringBuilder stringBuilder = new StringBuilder();
        for (int ii = asn1Encodables.length - 1; ii >= 0; --ii) {
            ASN1Encodable asn1Encodable = asn1Encodables[ii];
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int jj = 0; jj < dlSet.size(); ++jj) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(jj);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable attributeType = dlSequence.getObjectAt(0);
                ASN1Encodable attributeValue = dlSequence.getObjectAt(1);
                String string = DSSASN1Utils.getString(attributeValue);
                string = Rdn.escapeValue(string);
                if (stringBuilder.length() != 0) {
                    stringBuilder.append(',');
                }
                stringBuilder.append(attributeType).append('=').append(string);
            }
        }
        return stringBuilder.toString();
    }

    private static String getString(ASN1Encodable attributeValue) {
        String string;
        if (attributeValue instanceof ASN1String) {
            string = ((ASN1String)((Object)attributeValue)).getString();
        } else if (attributeValue instanceof ASN1ObjectIdentifier) {
            string = ((ASN1ObjectIdentifier)attributeValue).getId();
        } else {
            LOG.error("!!!*******!!! This encoding is unknown: {}", (Object)attributeValue.getClass().getSimpleName());
            string = attributeValue.toString();
            LOG.error("!!!*******!!! value: {}", (Object)string);
        }
        return string;
    }

    public static String extractAttributeFromX500Principal(ASN1ObjectIdentifier identifier, X500Principal x500PrincipalName) {
        RDN[] rdns;
        X500Name x500Name = X500Name.getInstance(x500PrincipalName.getEncoded());
        for (RDN rdn : rdns = x500Name.getRDNs(identifier)) {
            if (rdn.isMultiValued()) {
                AttributeTypeAndValue[] typesAndValues;
                for (AttributeTypeAndValue typeAndValue : typesAndValues = rdn.getTypesAndValues()) {
                    if (!identifier.equals(typeAndValue.getType())) continue;
                    return typeAndValue.getValue().toString();
                }
                continue;
            }
            AttributeTypeAndValue typeAndValue = rdn.getFirst();
            if (!identifier.equals(typeAndValue.getType())) continue;
            return typeAndValue.getValue().toString();
        }
        return null;
    }

    public static String getSubjectCommonName(CertificateToken cert) {
        return DSSASN1Utils.extractAttributeFromX500Principal(BCStyle.CN, cert.getSubjectX500Principal());
    }

    public static String getHumanReadableName(CertificateToken cert) {
        return DSSASN1Utils.firstNotNull(cert, BCStyle.CN, BCStyle.GIVENNAME, BCStyle.SURNAME, BCStyle.NAME, BCStyle.PSEUDONYM, BCStyle.O, BCStyle.OU);
    }

    private static String firstNotNull(CertificateToken cert, ASN1ObjectIdentifier ... oids) {
        for (ASN1ObjectIdentifier oid : oids) {
            String value = DSSASN1Utils.extractAttributeFromX500Principal(oid, cert.getSubjectX500Principal());
            if (value == null) continue;
            return value;
        }
        return null;
    }

    public static SignerInformation getFirstSignerInformation(CMSSignedData cms) {
        Collection<SignerInformation> signers = cms.getSignerInfos().getSigners();
        if (signers.size() > 1) {
            LOG.warn("!!! The framework handles only one signer (SignerInformation) !!!");
        }
        return signers.iterator().next();
    }

    public static boolean isASN1SequenceTag(byte tagByte) {
        return 48 == tagByte;
    }

    public static Date getDate(ASN1Encodable encodable) {
        try {
            return Time.getInstance(encodable).getDate();
        }
        catch (Exception e) {
            LOG.warn("Unable to retrieve the date : " + encodable, e);
            return null;
        }
    }

    public static boolean isEmpty(AttributeTable attributeTable) {
        return attributeTable == null || attributeTable.size() == 0;
    }

    public static AttributeTable emptyIfNull(AttributeTable original) {
        if (original == null) {
            return new AttributeTable(new Hashtable());
        }
        return original;
    }

    public static List<String> getExtendedKeyUsage(CertificateToken certToken) {
        try {
            return certToken.getCertificate().getExtendedKeyUsage();
        }
        catch (CertificateParsingException e) {
            LOG.warn("Unable to retrieve ExtendedKeyUsage : {}", (Object)e.getMessage());
            return Collections.emptyList();
        }
    }

    static {
        Security.addProvider(securityProvider);
    }
}

